package com.digiwin.mobile.mobileuibot.versionManage;

import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.model.db2.AppClient;
import com.digiwin.mobile.mobileuibot.model.db2.AppConfig;
import com.digiwin.mobile.mobileuibot.model.db2.AppUpdate;
import com.digiwin.mobile.mobileuibot.proxy.aim.service.DigiwinAimProxyService;
import com.digiwin.mobile.mobileuibot.proxy.cac.model.DigiwinCacCurrentUserApp;
import com.digiwin.mobile.mobileuibot.proxy.cac.service.DigiwinCacProxyService;
import com.digiwin.mobile.mobileuibot.versionManage.enums.*;
import com.digiwin.mobile.mobileuibot.versionManage.model.UpdateCondition;
import com.digiwin.mobile.mobileuibot.versionManage.model.UpdateConditionItem;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppUpgradeReq;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppCheckResp;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppUpgradeResp;
import com.digiwin.mobile.mobileuibot.versionManage.service.*;
import com.digiwin.mobile.mobileuibot.versionManage.utils.BrandIdentifier;
import com.digiwin.mobile.mobileuibot.versionManage.utils.VersionComparator;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>功能描述：版本升级服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionUpgradeServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/6/26 16:32
 */
@Service
@Slf4j
public class VersionUpgradeServiceImpl implements VersionUpgradeService {

    private static final List<String> ARCH_32 = Lists.newArrayList("armeabi-v7a");
    private static final List<String> ARCH_64 = Lists.newArrayList("arm64-v8a");

    @Autowired
    private AppConfigService appConfigService;
    @Autowired
    private AppClientService appClientService;
    @Autowired
    private AppUpdateService appUpdateService;
    @Autowired
    private AppConditionService appConditionService;
    @Autowired
    private AppConditionItemService appConditionItemService;
    @Autowired
    private AppUserClientRecordService appUserClientRecordService;

    @Autowired
    private DigiwinCacProxyService digiwinCacProxyService;

    @Autowired
    private DigiwinAimProxyService digiwinAimProxyService;

    @Override
    public AppUpgradeResp upgrade(AppUpgradeReq vo) {
        // wjw FIXME: 2025/6/27 异步记录用户客户端信息
        appUserClientRecordService.asyncStorage(vo);

        AppUpgradeResp result = new AppUpgradeResp().setUpdateStrategy(UpdateStrategyEnum.NONE.getValue());
        // 根据应用code，查询app_config
        AppConfig appConfig = appConfigService.getByCode(vo.getAppId());
        if (appConfig == null) {
            log.error("appConfig is null, code: {}", vo.getAppId());
            return result;
        }
        // 校验app_config是否有效
        if (!ValidEnum.VALID.getValue().equals(appConfig.getValid())) {
            log.error("appConfig is invalid, code: {}", vo.getAppId());
            return result;
        }
        // 根据appID获取已发布客户端数据（app_client） -- 当前校验的最新版本。
        // 注意：无已发布数据，直接返回 --- 因为如果根据中间有效的版本，最新的不确定是否可用（可能是配置错误了取消发布）
        AppClient latestAppClient = appClientService.getPublishByAppId(appConfig.getId());
        if (latestAppClient == null) {
            log.error("appClient is null, code: {}, appId: {}", vo.getAppId(), appConfig.getId());
            return result;
        }

        // 获取当前客户端版本更新配置
        UpdateStrategyEnum updateStrategy = this.getCurrentVersionUpdateStrategy(vo, appConfig, latestAppClient);

        // 构建返回结果
        return this.buildAppUpgradeResp(vo, result, appConfig, latestAppClient, updateStrategy);
    }

    /**
     * 获取当前客户端版本更新策略
     *
     * @param vo 客户端请求参数
     * @param appConfig 客户端配置信息
     * @param latestAppClient 最新客户端版本信息
     * @return 更新策略枚举
     */
    private UpdateStrategyEnum getCurrentVersionUpdateStrategy(AppUpgradeReq vo, AppConfig appConfig, AppClient latestAppClient) {
        // 判断当前客户端版本是否已是最新版本
        if (VersionComparator.compareVersions(vo.getVersion(), latestAppClient.getVersion()) >= 0) {
            log.error("client version is latest, code: {}, appId: {}, version: {}", vo.getAppId(), appConfig.getId(), vo.getVersion());
            // 已是最新版本
            return UpdateStrategyEnum.NONE;
        }

        // 获取当前客户端有效的版本信息
        AppClient cAppClient = appClientService.getValidByAppIdAndVersion(appConfig.getId(), vo.getVersion());
        if (cAppClient == null) {
            log.error("client version is not exist, code: {}, appId: {}, version: {}", vo.getAppId(), appConfig.getId(), vo.getVersion());
            // 非最新版本，当前客户端版本不存在，则获取最新客户端版本的更新策略
            AppUpdate latestAppUpdate = appUpdateService.getNotBakByAppClientId(latestAppClient.getId());
            Integer strongUpdate = Optional.ofNullable(latestAppUpdate).map(AppUpdate::getStrongUpdate).orElse(StrongUpdateEnum.NO.getValue());
            if (StrongUpdateEnum.YES.getValue().equals(strongUpdate)) {
                return UpdateStrategyEnum.FORCE;
            } else {
                return UpdateStrategyEnum.RECOMMEND;
            }
        }

        // wjw TODO: 2025/6/30 先拉取中间版本，获取中间版本是否有强更，有判断是否满足强更，满足则强更，不满足，则看是否满足提示更新
        // 检查中间版本是否有强更 --- 注意：包含了当前已发布的版本
        List<AppClient> middleAppClients = appClientService.getValidMiddleVersion(appConfig.getId(), latestAppClient.getSequence(), cAppClient.getSequence());
        if (CollectionUtils.isEmpty(middleAppClients)) {
            log.error("middle app clients is empty, code: {}, appId: {}, latestSequence: {}, sequence: {}", vo.getAppId(), appConfig.getId(), latestAppClient.getSequence(), cAppClient.getSequence());
            // 没有中间版本，则不提示
            return UpdateStrategyEnum.NONE;
        }
        Map<Long, AppClient> appClientMap = middleAppClients.stream().collect(Collectors.toMap(AppClient::getId, c -> c, (c1, c2) -> c1));
        List<AppUpdate> appUpdates = appUpdateService.getNotBakByAppClientIds(appClientMap.keySet());
        if (CollectionUtils.isEmpty(appUpdates)) {
            log.error("middle app updates is empty, code: {}, appId: {}, latestSequence: {}, sequence: {}, appClientIds: {}", vo.getAppId(), appConfig.getId(), latestAppClient.getSequence(), cAppClient.getSequence(), JsonUtil.javaObjectToJsonString(appClientMap.keySet()));
            // 没有中间版本更新配置，则不提示
            return UpdateStrategyEnum.NONE;
        }

        // 遍历中间版本更新配置，获取最高优先级的策略
        UpdateStrategyEnum updateStrategy = UpdateStrategyEnum.NONE;
        List<AppUpdate> strongAppUpdates = appUpdates.stream().filter(appUpdate -> StrongUpdateEnum.YES.getValue().equals(appUpdate.getStrongUpdate())).collect(Collectors.toList());
        for (AppUpdate appUpdate : strongAppUpdates) {
            AppClient appClient = appClientMap.get(appUpdate.getAppClientId());
            UpdateStrategyEnum strategy = this.checkUpdateStrategy(vo, appClient, appUpdate);
            if (UpdateStrategyEnum.FORCE == strategy) {
                log.error("exist force client version is force update, appClientId: {}, version: {}", appUpdate.getAppClientId(), Optional.ofNullable(appClient).map(AppClient::getVersion).orElse(null));
                // 中间版本有强更，且满足强更条件，则强制更新
                return strategy;
            }
            if (UpdateStrategyEnum.RECOMMEND == strategy) {
                log.error("exist force client version is recommend update, appClientId: {}, version: {}", appUpdate.getAppClientId(), Optional.ofNullable(appClient).map(AppClient::getVersion).orElse(null));
                // 中间版本有强更，且满足提示更新条件，则提示更新
                updateStrategy = strategy;
            }
        }
        if (UpdateStrategyEnum.NONE != updateStrategy) {
            // 中间版本有强更，且满足提示更新条件，则提示更新 -- 不需要判断非强更的中间版本
            return updateStrategy;
        }

        // 移除强更的中间版本
        appUpdates.removeAll(strongAppUpdates);
        for (AppUpdate appUpdate : appUpdates) {
            AppClient appClient = appClientMap.get(appUpdate.getAppClientId());
            UpdateStrategyEnum strategy = this.checkUpdateStrategy(vo, appClient, appUpdate);
            if (UpdateStrategyEnum.RECOMMEND == strategy) {
                log.error("exist recommend client version is recommend update, appClientId: {}, version: {}", appUpdate.getAppClientId(), Optional.ofNullable(appClient).map(AppClient::getVersion).orElse(null));
                // 中间版本有提示更新，且满足提示更新条件，则提示更新
                return strategy;
            }
        }
        return updateStrategy;
    }

    /**
     * 校验更新策略
     *
     * @param vo 客户端请求参数
     * @param appClient 客户端配置信息
     * @param appUpdate 客户端更新配置信息
     * @return 更新策略枚举
     */
    private UpdateStrategyEnum checkUpdateStrategy(AppUpgradeReq vo, AppClient appClient, AppUpdate appUpdate) {
        if (appClient == null || appUpdate == null) {
            return UpdateStrategyEnum.NONE;
        }
        if (ReleaseTypeEnum.FULL.getValue().equals(appUpdate.getReleaseType())) {
            // 全量发布 --- 全量发布下（未开启强更条件不可配置），开启强更，无条件全部强更，有条件不满足条件提示更新；关闭强更，没有条件都提示更新。
            if (StrongUpdateEnum.YES.getValue().equals(appUpdate.getStrongUpdate())) {
                // 强制更新
                if (UpdateRangeEnum.PARTIAL.getValue().equals(appUpdate.getUpdateRange())) {
                    // 部分用户
                    if (StringUtils.isBlank(appUpdate.getConditions())) {
                        // 无下发条件 -- 无条件全部提示更新。
                        return UpdateStrategyEnum.RECOMMEND;
                    }
                    if (this.checkConditionGroup(vo, appUpdate)) {
                        // 条件满足，强制更新
                        return UpdateStrategyEnum.FORCE;
                    } else {
                        // 条件不满足，提示更新
                        return UpdateStrategyEnum.RECOMMEND;
                    }
                } else {
                    // 全量用户 -- 无条件全部强制更新。
                    return UpdateStrategyEnum.FORCE;
                }
            } else {
                // 非强制更新 -- 无条件全部提示更新。
                return UpdateStrategyEnum.RECOMMEND;
            }
        } else {
            // 局部发布 --- 局部发布下(条件必须配置），满足条件提示更新（强更/提示），不满足条件的一律不提示。
            if (StringUtils.isBlank(appUpdate.getConditions())) {
                // 无下发条件
                return UpdateStrategyEnum.NONE;
            }
            if (StrongUpdateEnum.YES.getValue().equals(appUpdate.getStrongUpdate())) {
                // 强制更新
                if (this.checkConditionGroup(vo, appUpdate)) {
                    // 条件满足，强制更新
                    return UpdateStrategyEnum.FORCE;
                } else {
                    // 条件不满足，不更新
                    return UpdateStrategyEnum.NONE;
                }
            } else {
                // 非强制更新
                if (this.checkConditionGroup(vo, appUpdate)) {
                    // 条件满足，提示更新
                    return UpdateStrategyEnum.RECOMMEND;
                } else {
                    // 条件不满足，不更新
                    return UpdateStrategyEnum.NONE;
                }
            }
        }
    }

    /**
     * 校验条件组是否满足
     *
     * @param vo 客户端请求参数
     * @param appUpdate 客户端更新配置信息
     * @return 条件是否满足
     */
    private boolean checkConditionGroup(AppUpgradeReq vo, AppUpdate appUpdate) {
        String conditionStr = appUpdate.getConditions();
        if (StringUtils.isBlank(conditionStr)) {
            // 无下发条件
            return false;
        }

        List<UpdateCondition> conditionList;
        try {
            conditionList = JsonUtil.jsonStringToObject(conditionStr, new TypeReference<List<UpdateCondition>>() {
            });
        } catch (Exception e) {
            log.error("checkConditionGroup error, condition: {}", conditionStr, e);
            return false;
        }
        // 处理空列表
        if (CollectionUtils.isEmpty(conditionList)) {
            // 无下发条件
            return false;
        }

        boolean result = false;
        // 遍历条件组，判断是否满足
        for (int i = 0; i < conditionList.size(); i++) {
            UpdateCondition curr = conditionList.get(i);
            if (i == 0) {
                // 第一个条件
                result = this.checkCondition(vo, curr);
            } else {
                // 非第一个条件，根据上一个条件的逻辑判断
                UpdateCondition prev = conditionList.get(i - 1);
                if ("OR".equalsIgnoreCase(prev.getLogic())) {
                    // OR
                    // 上一个条件逻辑为OR(表示上一个条件result=false）
                    result = this.checkCondition(vo, curr);
                } else {
                    // AND
                    result = result ? this.checkCondition(vo, curr) : result;
                }
            }
            // 当前条件逻辑为OR，且为true，则返回
            if ("OR".equalsIgnoreCase(curr.getLogic()) && result) {
                return true;
            }
        }
        return result;
    }

    /**
     * 校验条件是否满足
     *
     * @param vo 升级请求参数
     * @param condition 条件
     * @return 条件是否满足
     */
    private boolean checkCondition(AppUpgradeReq vo, UpdateCondition condition) {
        if (condition == null) {
            return false;
        }
        List<UpdateConditionItem> item = condition.getItem();
        if (CollectionUtils.isEmpty(item)) {
            return false;
        }
        boolean result = false;
        // 遍历条件组，判断是否满足
        for (int i = 0; i < item.size(); i++) {
            UpdateConditionItem curr = item.get(i);
            if (i == 0) {
                // 第一个条件
                result = this.checkConditionItem(vo, curr);
            } else {
                // 非第一个条件，根据上一个条件的逻辑判断
                UpdateConditionItem prev = item.get(i - 1);
                if ("OR".equalsIgnoreCase(prev.getLogic())) {
                    // OR
                    // 上一个条件逻辑为OR(表示上一个条件result=false）
                    result = this.checkConditionItem(vo, curr);
                } else {
                    // AND
                    result = result ? this.checkConditionItem(vo, curr) : result;
                }
            }
            // 当前条件逻辑为OR，且为true，则返回
            if ("OR".equalsIgnoreCase(curr.getLogic()) && result) {
                return true;
            }
        }
        return result;
    }

    /**
     * 校验条件项是否满足
     *
     * @param vo 升级请求参数
     * @param curr 条件项
     * @return 条件项是否满足
     */
    private boolean checkConditionItem(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        // 获取组件枚举
        ComponentEnum componentEnum = ComponentEnum.exist(curr.getComponent());
        if (componentEnum == null) {
            return false;
        }
        boolean result;
        // 处理组件
        switch (componentEnum) {
            case SINGLE_TEXT:
            case MULTI_TEXT:
                result = this.checkText(vo, curr);
                break;
            case SELECT_SINGLE:
            case SELECT_MULTI:
                result = this.checkSelect(vo, curr);
                break;
            case SELECT_TENANT:
                result = this.checkSelectTenant(vo, curr);
                break;
            default:
                throw new IllegalArgumentException("unknown component");
        }
        return result;
    }

    /**
     * 校验文本是否满足
     *
     * @param vo 升级请求参数
     * @param curr 条件项
     * @return 条件项是否满足
     */
    private boolean checkText(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        // 判断购买的商品
        if ("goods_code".equalsIgnoreCase(curr.getSchema())) {
            return this.checkGoodsCode(vo, curr);
        }
        // 判断常规的文本
        return this.checkCommonText(vo, curr);
    }

    /**
     * 校验常规文本是否满足
     *
     * @param vo 升级请求参数
     * @param curr 条件项
     * @return 条件项是否满足
     */
    private boolean checkCommonText(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        List<String> serverValue = curr.getValue();
        if (CollectionUtils.isEmpty(serverValue)) {
            return false;
        }
        Map<String, Object> rawMap = JsonUtil.objectToJavaObject(vo, new TypeReference<Map<String, Object>>() {
        });
        String clientValue = StringUtil.valueOf(rawMap.get(curr.getSchema())).toLowerCase();
        if (StringUtils.isBlank(clientValue)) {
            return false;
        }
        String serverValueStr = serverValue.get(0).toLowerCase();
        if (serverValueStr.contains(",")) {
            // 多个值 -- 操作符默认是等于 -- 后期有场景在扩展
            String[] serverValueArr = serverValueStr.split(",");
            return Arrays.asList(serverValueArr).contains(clientValue);
        }
        // 单个值
        OperatorEnum operatorEnum = OperatorEnum.exist(curr.getOperator());
        if (operatorEnum == null) {
            // 未知操作符 -- 默认是等于
            return serverValueStr.equalsIgnoreCase(clientValue);
        }
        // 处理操作符
        switch (operatorEnum) {
            case EQ:
                return serverValueStr.equalsIgnoreCase(clientValue);
            case LT:
                return Integer.parseInt(clientValue) < Integer.parseInt(serverValueStr);
            case LE:
                return Integer.parseInt(clientValue) <= Integer.parseInt(serverValueStr);
            case GT:
                return Integer.parseInt(clientValue) > Integer.parseInt(serverValueStr);
            case GE:
                return Integer.parseInt(clientValue) >= Integer.parseInt(serverValueStr);
            default:
                // 未知操作符 -- 默认是等于
                return serverValueStr.equalsIgnoreCase(clientValue);
        }
    }

    /**
     * 校验商品码是否存在
     *
     * @param vo          升级请求参数
     * @param curr         条件项
     * @return 升级条件项是否满足
     */
    private boolean checkGoodsCode(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        List<String> serverValue = curr.getValue();
        if (CollectionUtils.isEmpty(serverValue)) {
            return false;
        }
        // 判断购买的商品
        if (StringUtils.isBlank(vo.getUserId()) || StringUtils.isBlank(vo.getIamUserToken())) {
            // 说明未登录下触发版更接口
            return false;
        }
        // 区分大小写
        String serverValueStr = serverValue.get(0);
        List<String> serverValueList = Arrays.stream(serverValueStr.contains(",") ? serverValueStr.split(",") : new String[]{serverValueStr}).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        List<DigiwinCacCurrentUserApp> authApps = digiwinCacProxyService.findAuthorizationsCurrentUserApp(vo.getUserId(), vo.getIamUserToken());
        if (CollectionUtils.isEmpty(authApps)) {
            // 无授权的商品
            return false;
        }
        // 遍历授权商品，判断是否包含购买的商品
        return authApps.stream().anyMatch(app -> serverValueList.contains(app.getId()) && BooleanUtils.isTrue(DateTimeUtil.isAfterNowDatetime(app.getExpiredTime())));

    }

    /**
     * 校验下拉选择框是否满足
     *
     * @param vo 升级请求参数
     * @param curr 条件项
     * @return 条件项是否满足
     */
    private boolean checkSelect(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        List<String> serverValue = curr.getValue();
        if (CollectionUtils.isEmpty(serverValue)) {
            return false;
        }
        // 转小写
        serverValue = serverValue.stream().map(String::toLowerCase).filter(StringUtils::isNotBlank).collect(Collectors.toList());

        Map<String, Object> rawMap = JsonUtil.objectToJavaObject(vo, new TypeReference<Map<String, Object>>() {
        });
        String clientValue = StringUtil.valueOf(rawMap.get(curr.getSchema())).toLowerCase();
        if (StringUtils.isBlank(clientValue)) {
            return false;
        }
        // 下拉选择框 -- 操作符默认是等于 -- 后期有场景在扩展
        return serverValue.contains(clientValue);
    }

    /**
     * 校验租户下拉选择框是否满足
     *
     * @param vo 升级请求参数
     * @param curr 条件项
     * @return 条件项是否满足
     */
    private boolean checkSelectTenant(AppUpgradeReq vo, UpdateConditionItem curr) {
        if (curr == null) {
            return false;
        }
        if (StringUtils.isBlank(vo.getUserId()) || StringUtils.isBlank(vo.getIamUserToken()) || StringUtils.isBlank(vo.getTenantId())) {
            // 说明未登录下触发版更接口
            return false;
        }

        boolean result = false;
        List<String> serverValue = curr.getValue();
        if (CollectionUtils.isNotEmpty(serverValue)) {
            // 其它租户是否满足
            result = serverValue.contains(vo.getTenantId());
        }
        if (result) {
            return result;
        }
        if (BooleanUtils.isNotTrue(curr.getOpsTenant())) {
            return result;
        }
        // 校验是否是ops预区租户 (PRE：预区。PROD：正式区)
        String tenantRouteType = digiwinAimProxyService.getTenantRouteType(vo.getIamUserToken());
        return "PRE".equalsIgnoreCase(tenantRouteType);
    }

    /**
     * 构建返回结果
     *
     * @param vo 升级请求参数
     * @param result 返回结果
     * @param appConfig 应用配置信息
     * @param latestAppClient 最新客户端信息
     * @param updateStrategy 更新策略枚举
     * @return 返回结果
     */
    private AppUpgradeResp buildAppUpgradeResp(AppUpgradeReq vo, AppUpgradeResp result, AppConfig appConfig, AppClient latestAppClient, UpdateStrategyEnum updateStrategy) {
        if (UpdateStrategyEnum.NONE == updateStrategy) {
            // 无更新策略
            return result;
        }
        result.setLatestVersion(latestAppClient.getVersion())
                .setUpdateStrategy(updateStrategy.getValue())
                .setMd5(this.getMd5(vo.getArch(), appConfig.getOsType(), latestAppClient))
                .setStore(this.isStore(vo.getBrand(), appConfig.getOsType(), latestAppClient))
                .setUrl(this.getUrl(vo.getArch(), appConfig.getOsType(), latestAppClient))
                .setTitle(latestAppClient.getTipsTitle(vo.getLocale()))
                .setTips(latestAppClient.getTips(vo.getLocale()));
        return result;
    }

    /**
     * 获取md5
     *
     * @param arch 架构
     * @param osType 系统类型android/ios
     * @param latestAppClient 最新客户端信息
     * @return md5
     */
    private String getMd5(String arch, String osType, AppClient latestAppClient) {
        if ("ios".equalsIgnoreCase(osType)) {
            // ios 不存在md5值
            return null;
        }
        // 默认下载32位版本
        String md5 = latestAppClient.getMd532();
        if (ARCH_64.contains(arch)) {
            md5 = latestAppClient.getMd564();
        }
        return md5;
    }

    /**
     * 获取下载地址
     *
     * @param arch 架构
     * @param osType 系统类型android/ios
     * @param latestAppClient 最新客户端信息
     * @return 下载地址
     */
    private String getUrl(String arch, String osType, AppClient latestAppClient) {
        // 默认下载32位版本
        String url = latestAppClient.getUrl32();
        if ("ios".equalsIgnoreCase(osType)) {
            // ios 下载地址存储在url32
            return url;
        }
        if (ARCH_64.contains(arch)) {
            url = latestAppClient.getUrl64();
        }
        return url;
    }


    /**
     * 是否跳转到应用商店
     *
     * @param inputBrand      前端传递的品牌
     * @param osType          系统类型android/ios
     * @param latestAppClient 最新客户端信息
     * @return 是否跳转到应用商店
     */
    private Boolean isStore(String inputBrand, String osType, AppClient latestAppClient) {
        if ("ios".equalsIgnoreCase(osType)) {
            // ios 默认跳转商店
            return true;
        }
        if (StringUtils.isBlank(inputBrand) || StringUtils.isBlank(latestAppClient.getBrand())) {
            return false;
        }
        // 前端传递的品牌
        BrandEnum brandEnum = BrandEnum.exist(BrandIdentifier.identifyBrand(inputBrand));
        if (brandEnum == null) {
            return false;
        }
        return Arrays.stream(latestAppClient.getBrand().split(","))
                .map(String::trim).filter(StringUtils::isNotBlank)
                .anyMatch(r -> r.equalsIgnoreCase(brandEnum.getValue()));
    }

    @Override
    public AppCheckResp checkUpdate(AppUpgradeReq vo) {
        // 默认是最新版本且不更新
        AppCheckResp result = (AppCheckResp) new AppCheckResp().setIsLatest(true).setUpdateStrategy(UpdateStrategyEnum.NONE.getValue());
        // 根据应用code，查询app_config
        AppConfig appConfig = appConfigService.getByCode(vo.getAppId());
        if (appConfig == null) {
            log.error("appConfig is null, code: {}", vo.getAppId());
            return result;
        }
        // 校验app_config是否有效
        if (!ValidEnum.VALID.getValue().equals(appConfig.getValid())) {
            log.error("appConfig is invalid, code: {}", vo.getAppId());
            return result;
        }
        // 根据appID获取已发布客户端数据（app_client） -- 当前校验的最新版本。
        // 注意：无已发布数据，直接返回 --- 因为如果根据中间有效的版本，最新的不确定是否可用（可能是配置错误了取消发布）
        AppClient latestAppClient = appClientService.getPublishByAppId(appConfig.getId());
        if (latestAppClient == null) {
            log.error("appClient is null, code: {}, appId: {}", vo.getAppId(), appConfig.getId());
            return result;
        }
        AppUpdate latestAppUpdate = appUpdateService.getNotBakByAppClientId(latestAppClient.getId());
        if (latestAppUpdate == null) {
            log.error("appUpdate is null, code: {}, appId: {}, appClientId: {}", vo.getAppId(), appConfig.getId(), latestAppClient.getId());
            return result;
        }

        // 是否最新版本
        result.setIsLatest(VersionComparator.compareVersions(vo.getVersion(), latestAppClient.getVersion()) >= 0);
        if (result.getIsLatest()) {
            // 已是最新版本
            return (AppCheckResp) result.setLatestVersion(latestAppClient.getVersion());
        }
        // 构建返回结果
        return (AppCheckResp) this.buildAppUpgradeResp(vo, result, appConfig, latestAppClient, UpdateStrategyEnum.RECOMMEND);
    }
}
