package com.digiwin.mobile.mobileuibot.versionManage.request;

import com.digiwin.mobile.mobileuibot.api.EnumConstraint;
import com.digiwin.mobile.mobileuibot.api.EnumIntegerConstraint;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.versionManage.enums.ComponentEnum;
import com.digiwin.mobile.mobileuibot.versionManage.model.ConditionItemSelectContent;
import com.digiwin.mobile.mobileuibot.versionManage.model.ConditionItemTextVerify;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.Serializable;

/**
 * <p>功能描述：条件项请求对象</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AppConditionItemRequest.java
 * @Author: wangjwc
 * @Date: created at 2025/8/11 16:59
 */
@Data
public class AppConditionItemRequest implements Serializable {
    private static final long serialVersionUID = -5011713046645095957L;
    /**
     * 主键id
     */
    private Long id;
    /**
     * 条件ID（关联app_condition.id）
     */
    private Long appConditionId;
    /**
     * schema--需要与端上约定（与入参一致）
     */
    @NotBlank(message = "schema不能为空")
    private String schema;
    /**
     * 名称
     */
    @NotBlank(message = "条件项名称不能为空")
    private String name;
    /**
     * 条件符：0不显示，1显示
     */
    @NotNull(message = "条件符不能为空")
    @EnumIntegerConstraint(value = {0, 1}, message = "条件符值无效")
    private Integer symbol;
    /**
     * 条件值组件：0单行文本，1多行文本，2下拉单选，3下拉多选，4租户选择框
     * @see ComponentEnum
     */
    @NotNull(message = "条件值组件不能为空")
    @EnumConstraint(
            enumClass = ComponentEnum.class,
            valueMethod = "getValue",
            message = "字段 component 的输入值 '{value}' 不合法! 有效选项: {values}")
    private Integer component;
    /**
     * 条件值，component=2/3
     * ConditionItemSelectContent
     *  {
     *         "options": [
     *                   {
     *                     "id": "android",
     *                     "text": "Android"
     *                   },
     *                   {
     *                     "id": "ios",
     *                     "text": "iOS"
     *                   }
     *             ]
     * }
     */
    private Object content;
    /**
     * 条件值校验，component=0/1
     * ConditionItemTextVerify
     * {
     *         "msg": "" , // 文案
     *         "script": "" // 效验脚本
     * }
     */
    private Object verify;

    /**
     * 验证content参数
     * @return true/false
     */
    public boolean verifyContentParam() {
        // 条件值，component=2/3
        if (ComponentEnum.SELECT_SINGLE.getValue().equals(this.component) || ComponentEnum.SELECT_MULTI.getValue().equals(this.component)) {
            // content不能为空
            if (this.content == null || StringUtils.isBlank(this.content.toString())) {
                return false;
            }
            try {
                JsonUtil.objectToJavaObject(this.content, ConditionItemSelectContent.class);
            } catch (Exception e) {
                return false;
            }
        }
        return true;
    }

    /**
     * 验证verify参数
     * @return true/false
     */
    public boolean verifyVerifyParam() {
        // 条件值校验，component=0/1
        if (ComponentEnum.SINGLE_TEXT.getValue().equals(this.component) || ComponentEnum.MULTI_TEXT.getValue().equals(this.component)) {
            // verify 可以为空
            if (this.verify == null || StringUtils.isBlank(this.verify.toString())) {
                return true;
            }
            try {
                JsonUtil.objectToJavaObject(this.verify, ConditionItemTextVerify.class);
            } catch (Exception e) {
                return false;
            }
        }
        return true;
    }

    /**
     * 转换content参数
     * @return 转换后的json字符串
     */
    public String contentCovert() {
        if (this.content == null || StringUtils.isBlank(this.content.toString())) {
            return "";
        }
        // 条件值，component=2/3
        if (ComponentEnum.SELECT_SINGLE.getValue().equals(this.component) || ComponentEnum.SELECT_MULTI.getValue().equals(this.component)) {
            return JsonUtil.javaObjectToJsonString(this.content);
        }
        return "";
    }

    /**
     * 转换verify参数
     * @return 转换后的json字符串
     */
    public String verifyCovert() {
        if (this.verify == null || StringUtils.isBlank(this.verify.toString())) {
            return "";
        }
        // 条件值校验，component=0/1
        if (ComponentEnum.SINGLE_TEXT.getValue().equals(this.component) || ComponentEnum.MULTI_TEXT.getValue().equals(this.component)) {
            return JsonUtil.javaObjectToJsonString(this.verify);
        }
        return "";
    }

}
