package com.digiwin.mobile.mobileuibot.versionManage.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.mobile.mobileuibot.dto.AppClientPageDto;
import com.digiwin.mobile.mobileuibot.dto.AppClientVo;
import com.digiwin.mobile.mobileuibot.mapper.db2.AppClientMapper;
import com.digiwin.mobile.mobileuibot.mock.enums.DelFlagEnum;
import com.digiwin.mobile.mobileuibot.model.db2.AppClient;
import com.digiwin.mobile.mobileuibot.versionManage.enums.PublishEnum;
import com.digiwin.mobile.mobileuibot.versionManage.enums.ValidEnum;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientUpdateRequest;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppClientService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;

/**
 * <p>功能描述：客户端版本配置 服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AppClientServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/6/27 10:17
 */
@Service
public class AppClientServiceImpl implements AppClientService {
    @Autowired
    private AppClientMapper appClientMapper;

    @Override
    public AppClient getPublishByAppId(Long appId) {
        if (appId == null) {
            return null;
        }
        LambdaQueryWrapper<AppClient> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AppClient::getAppId, appId);
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        wrapper.eq(AppClient::getPublish, PublishEnum.PUBLISH.getValue());
        return appClientMapper.selectOne(wrapper, false);
    }

    @Override
    public AppClient getValidByAppIdAndVersion(Long appId, String version) {
        if (appId == null || StringUtils.isBlank(version)) {
            return null;
        }
        LambdaQueryWrapper<AppClient> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AppClient::getAppId, appId);
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        wrapper.eq(AppClient::getVersion, version);
        wrapper.eq(AppClient::getValid, ValidEnum.VALID.getValue());
        return appClientMapper.selectOne(wrapper, false);
    }

    @Override
    public List<AppClient> getValidMiddleVersion(Long appId, Integer latestSequence, Integer sequence) {
        if (appId == null || latestSequence == null || sequence == null) {
            return Collections.emptyList();
        }
        LambdaQueryWrapper<AppClient> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AppClient::getAppId, appId);
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        wrapper.eq(AppClient::getValid, ValidEnum.VALID.getValue());
        // sequence < 版本 <= latestSequence
        wrapper.gt(AppClient::getSequence, sequence).le(AppClient::getSequence, latestSequence);
        // 降序 -- 最新版本在最前
        wrapper.orderByDesc(AppClient::getSequence);
        return appClientMapper.selectList(wrapper);
    }

    @Override
    public Page<AppClientVo> listPage(AppClientPageDto pageRequest) {
        if (pageRequest == null) {
            return new Page<>();
        }
        // 1. 创建分页对象（当前页, 每页数量）
        Page<AppClientVo> page = new Page<>(pageRequest.getPageNum(), pageRequest.getPageSize());
        // 2. 执行多表联合查询
        return appClientMapper.listPage(page, pageRequest);
    }

    @Override
    public AppClientVo queryById(Long id) {
        if (id == null) {
            return null;
        }
        return appClientMapper.queryById(id);
    }

    @Override
    public AppClient selectById(Long id) {
        if (id == null) {
            return null;
        }
        LambdaQueryWrapper<AppClient> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AppClient::getId, id);
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        return appClientMapper.selectOne(wrapper, false);
    }

    @Override
    public AppClient add(AppClientAddRequest addRequest) {
        if (addRequest == null) {
            return null;
        }
        AppClient appClient = new AppClient()
                .setId(null)
                .setAppId(addRequest.getAppId())
                .setVersion(addRequest.getVersion())
                .setSequence(this.getLatestSequence(addRequest.getAppId(), addRequest.getVersion()))
                .setDescr(addRequest.getDescr())
                .setMethods(addRequest.getMethods())
                .setUrl32(addRequest.getUrl32())
                .setUrl64(addRequest.getUrl64())
                .setMd532(addRequest.getMd532())
                .setMd564(addRequest.getMd564())
                .setBrand(addRequest.convertBrand())
                .setTipsTitle(addRequest.getTipsTitle())
                .setTips(addRequest.getTips())
                .setLang(addRequest.convertLang())
                // 无效、未发布、未删除
                .setValid(ValidEnum.INVALID.getValue())
                .setPublish(PublishEnum.UNPUBLISH.getValue())
                .setDelFlag(DelFlagEnum.UNDELETE.getValue());
        appClientMapper.insert(appClient);
        return appClient;
    }

    /**
     * 获取sequence值
     * sequence值，同一个app_id、version下sequence值一样（需考虑删除的数据）
     * 1、新增客户端数据时，查询当前app_id、version下数据是否存在。（需考虑删除的数据）
     * 1.1、存在获取sequence值赋予当前新增数据sequence值
     * 1.2、不存在，获取app_id下sequence最大值 +1 赋予当前新增数据的sequence值（需考虑删除的数据）
     *
     * @param appId 应用id
     * @param version 版本号
     * @return sequence值
     */
    private Integer getLatestSequence(Long appId, String version) {
        if (appId == null || StringUtils.isBlank(version)) {
            throw new IllegalArgumentException("appId or version is null or empty");
        }
        // 1、新增客户端数据时，查询当前app_id、version下数据是否存在（需考虑删除的数据）
        LambdaQueryWrapper<AppClient> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(AppClient::getAppId, appId);
        wrapper.eq(AppClient::getVersion, version);
        AppClient appClient = appClientMapper.selectOne(wrapper, false);
        // 1.1、存在获取sequence值赋予当前新增数据sequence值
        if (appClient != null) {
            return appClient.getSequence();
        }
        // 1.2、不存在，获取app_id下sequence最大值 +1 赋予当前新增数据的sequence值（需考虑删除的数据）
        LambdaQueryWrapper<AppClient> wrapper2 = new LambdaQueryWrapper<>();
        wrapper2.eq(AppClient::getAppId, appId);
        wrapper2.orderByDesc(AppClient::getSequence);
        AppClient appClient2 = appClientMapper.selectOne(wrapper2, false);
        return appClient2 == null ? 1 : appClient2.getSequence() + 1;
    }

    @Override
    public void update(AppClient appClient, AppClientUpdateRequest updateRequest) {
        if (appClient == null || updateRequest == null) {
            return;
        }
        // appId、version、sequence不能修改
        appClient.setDescr(updateRequest.getDescr())
                .setMethods(updateRequest.getMethods())
                .setUrl32(updateRequest.getUrl32())
                .setUrl64(updateRequest.getUrl64())
                .setMd532(updateRequest.getMd532())
                .setMd564(updateRequest.getMd564())
                .setBrand(updateRequest.convertBrand())
                .setTipsTitle(updateRequest.getTipsTitle())
                .setTips(updateRequest.getTips())
                .setLang(updateRequest.convertLang())
                // 清空创建时间和更新时间，走数据库默认值
                .clearTime();
        appClientMapper.updateById(appClient);
    }

    @Override
    public void release(AppClient appClient, AppClientUpdateRequest releaseRequest) {
        if (appClient == null || releaseRequest == null) {
            return;
        }
        // appId、version、sequence不能修改
        appClient.setDescr(releaseRequest.getDescr())
                .setMethods(releaseRequest.getMethods())
                .setUrl32(releaseRequest.getUrl32())
                .setUrl64(releaseRequest.getUrl64())
                .setMd532(releaseRequest.getMd532())
                .setMd564(releaseRequest.getMd564())
                .setBrand(releaseRequest.convertBrand())
                .setTipsTitle(releaseRequest.getTipsTitle())
                .setTips(releaseRequest.getTips())
                .setLang(releaseRequest.convertLang())
                // 有效、已发布、未删除
                .setValid(ValidEnum.VALID.getValue())
                .setPublish(PublishEnum.PUBLISH.getValue())
                .setDelFlag(DelFlagEnum.UNDELETE.getValue())
                // 清空创建时间和更新时间，走数据库默认值
                .clearTime();
        appClientMapper.updateById(appClient);

        // 同一个app_id下只有一个已发布，其它设置为取消发布
        LambdaUpdateWrapper<AppClient> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(AppClient::getAppId, appClient.getAppId());
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        wrapper.eq(AppClient::getPublish, PublishEnum.PUBLISH.getValue());
        wrapper.ne(AppClient::getId, appClient.getId());
        wrapper.set(AppClient::getPublish, PublishEnum.CANCELPUBLISH.getValue());
        appClientMapper.update(null, wrapper);

        // 注意：同一个app_id与version下，只有一个有效(点击发布按钮保存的数据设置为有效，其它设置为无效)
        LambdaUpdateWrapper<AppClient> wrapper2 = new LambdaUpdateWrapper<>();
        wrapper2.eq(AppClient::getAppId, appClient.getAppId());
        wrapper2.eq(AppClient::getVersion, appClient.getVersion());
        wrapper2.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());
        wrapper2.eq(AppClient::getValid, ValidEnum.VALID.getValue());
        wrapper2.ne(AppClient::getId, appClient.getId());
        wrapper2.set(AppClient::getValid, ValidEnum.INVALID.getValue());
        appClientMapper.update(null, wrapper2);
    }

    @Override
    public void unrelease(AppClient appClient) {
        if (appClient == null) {
            return;
        }
        // 取消发布
        appClient.setPublish(PublishEnum.CANCELPUBLISH.getValue())
                // 清空创建时间和更新时间，走数据库默认值
                .clearTime();
        appClientMapper.updateById(appClient);
    }

    @Override
    public void deleteById(Long id) {
        if (id == null) {
            return;
        }
        // 删除
        LambdaUpdateWrapper<AppClient> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(AppClient::getId, id);
        wrapper.eq(AppClient::getDelFlag, DelFlagEnum.UNDELETE.getValue());

        wrapper.set(AppClient::getDelFlag, DelFlagEnum.DELETE.getValue());
        appClientMapper.update(null, wrapper);
    }
}