package com.digiwin.mobile.mobileuibot.versionManage.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.mapper.db2.AppUserClientRecordMapper;
import com.digiwin.mobile.mobileuibot.model.db2.AppUserClientRecord;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppUpgradeReq;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppUserClientRecordService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;

/**
 * <p>功能描述：app用户客户端记录 服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AppUserClientRecordServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/6/27 10:17
 */
@Service
@Slf4j
public class AppUserClientRecordServiceImpl implements AppUserClientRecordService {
    /**
     * 缓存内容，默认容量 1000
     */
    public static BlockingDeque<AppUserClientRecord> BLOCKING_DEQUE = new LinkedBlockingDeque<>(1000);

    private final AppUserClientRecordMapper appUserClientRecordMapper;

    @Autowired
    public AppUserClientRecordServiceImpl(AppUserClientRecordMapper appUserClientRecordMapper) {
        this.appUserClientRecordMapper = appUserClientRecordMapper;
        this.initConsumer();
    }

    /**
     * 初始化日志消费者线程
     */
    private void initConsumer() {
        Thread consumerThread = new Thread(() -> {
            while (true) {
                try {
                    // 休眠50毫秒,防止CPU空转
                    Thread.sleep(50);

                    // 从队列中取日志（带超时）
                    AppUserClientRecord appUserClientRecord = BLOCKING_DEQUE.poll(500, TimeUnit.MILLISECONDS);
                    if (appUserClientRecord == null) {
                        continue;
                    }

                    // 查询是否存在相同记录, 如果存在则更新，不存在则插入
                    LambdaQueryWrapper<AppUserClientRecord> wrapper = new LambdaQueryWrapper<>();
                    wrapper.eq(AppUserClientRecord::getUserId, appUserClientRecord.getUserId());
                    wrapper.eq(AppUserClientRecord::getDeviceId, appUserClientRecord.getDeviceId());
                    if (appUserClientRecordMapper.exists(wrapper)) {
                        // entity中属性值为null，会忽略该属性
                        appUserClientRecordMapper.update(appUserClientRecord, wrapper);
                    } else {
                        appUserClientRecordMapper.insert(appUserClientRecord);
                    }
                } catch (Exception e) {
                    log.error("app用户客户端记录 保存异常", e);
                    // 异常处理：重试或记录错误日志
                }
            }
        }, "app_user_client_record-Consumer");
        consumerThread.setDaemon(true);
        consumerThread.start();
    }


    @Override
    public void asyncStorage(AppUpgradeReq vo) {
        try {
            if (StringUtils.isBlank(vo.getUserId())) {
                // 说明未登录下触发版更接口，不记录日志
                return;
            }
            AppUserClientRecord appUserClientRecord = JsonUtil.objectToJavaObject(vo, AppUserClientRecord.class);
            // 添加日志到缓冲队列
            if (!BLOCKING_DEQUE.offer(appUserClientRecord)) {
                log.warn("app用户客户端记录 队列已满，丢弃日志: {}", JsonUtil.javaObjectToJsonString(appUserClientRecord));
                // 可添加降级策略：写入文件或发送告警
            }
        } catch (Exception e) {
            log.error("app用户客户端记录 转换异常", e);
        }
    }
}