package com.digiwin.mobile.mobileuibot.versionManage.utils;

import com.digiwin.mobile.mobileuibot.versionManage.enums.BrandEnum;

import java.util.*;
import java.util.regex.Pattern;

/**
 * <p>功能描述：品牌识别工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: BrandIdentifier.java
 * @Author: wangjwc
 * @Date: created at 2025/6/30 21:00
 */
public class BrandIdentifier {

    // 品牌映射表（主品牌 -> 子品牌/别名）
    private static final Map<BrandEnum, Set<String>> BRAND_MAPPING = new HashMap<>();

    // 特殊品牌映射（优先级高于常规匹配）
    private static final Map<Pattern, BrandEnum> SPECIAL_PATTERNS = new HashMap<>();

    // 型号关键词映射
    private static final Map<Pattern, BrandEnum> MODEL_KEYWORDS = new HashMap<>();

    // 构建 Trie 树
    private static final Trie PREFIX_TRIE = new Trie();

    static {
        // 初始化特殊模式匹配
        SPECIAL_PATTERNS.put(Pattern.compile("^(redmi|red\\s?mi|红米)"), BrandEnum.XIAOMI);
        SPECIAL_PATTERNS.put(Pattern.compile("^(oneplus|one\\s?plus|一加)"), BrandEnum.OPPO);

        // 精确别名匹配
        // 小米集团
        Set<String> xiaomiGroup = new HashSet<>(Arrays.asList(
                "xiaomi", "mi", "redmi", "poco", "blackshark", "hmd"
        ));
        BRAND_MAPPING.put(BrandEnum.XIAOMI, xiaomiGroup);

        // OPPO集团
        Set<String> oppoGroup = new HashSet<>(Arrays.asList(
                "oppo", "realme", "oneplus", "one plus"
        ));
        BRAND_MAPPING.put(BrandEnum.OPPO, oppoGroup);

        // 华为集团
        Set<String> huaweiGroup = new HashSet<>(Arrays.asList(
                "huawei", "honor", "hua wei"
        ));
        BRAND_MAPPING.put(BrandEnum.HUAWEI, huaweiGroup);

        // VIVO集团
        Set<String> vivoGroup = new HashSet<>(Arrays.asList(
                "vivo"
        ));
        BRAND_MAPPING.put(BrandEnum.VIVO, vivoGroup);

        // 构建 Trie 树
        for (Map.Entry<BrandEnum, Set<String>> entry : BRAND_MAPPING.entrySet()) {
            for (String alias : entry.getValue()) {
                PREFIX_TRIE.insert(alias, entry.getKey());
            }
        }

        // 初始化型号关键词
        MODEL_KEYWORDS.put(Pattern.compile(".*(redmi|mi\\s?\\d+|poco).*"), BrandEnum.XIAOMI);
        MODEL_KEYWORDS.put(Pattern.compile(".*(realme|oneplus|oppo\\s?reno).*"), BrandEnum.OPPO);
        MODEL_KEYWORDS.put(Pattern.compile(".*(honor|huawei\\s?mate|huawei\\s?p\\d+).*"), BrandEnum.HUAWEI);
    }

    /**
     * 标准化品牌名称
     * // 识别优先级：
     * // 1. 特殊模式匹配（正则）
     * // 2. 精确别名匹配
     * // 3. 模糊匹配（Levenshtein距离）
     * // 4. 型号关键词匹配（正则）
     * // 5. 数字型号特征匹配
     * // 6. 返回Unknown
     * @param inputBrand 前端传入的品牌字符串
     * @return 标准化的品牌名称，未知品牌返回 null
     */
    public static String identifyBrand(String inputBrand) {
        if (inputBrand == null || inputBrand.trim().isEmpty()) {
            return null;
        }

        // 清洗输入：转小写、移除空格/连字符
        String cleaned = cleanInput(inputBrand);

        // 1. 特殊模式匹配（高优先级）
        for (Map.Entry<Pattern, BrandEnum> entry : SPECIAL_PATTERNS.entrySet()) {
            if (entry.getKey().matcher(cleaned).find()) {
                return entry.getValue().getValue();
            }
        }

        // 2. 精确匹配品牌别名
        for (Map.Entry<BrandEnum, Set<String>> entry : BRAND_MAPPING.entrySet()) {
            if (entry.getValue().contains(cleaned)) {
                return entry.getKey().getValue();
            }
        }

        // 3. 模糊匹配（处理拼写错误）
        // 替代模糊匹配
        BrandEnum matchedBrand = PREFIX_TRIE.searchPrefix(cleaned);
        if (matchedBrand != null) {
            return matchedBrand.getValue();
        }

        // 4. 型号关键词匹配
        for (Map.Entry<Pattern, BrandEnum> entry : MODEL_KEYWORDS.entrySet()) {
            if (entry.getKey().matcher(cleaned).find()) {
                return entry.getValue().getValue();
            }
        }

        // 5. 数字型号特征匹配
        if (cleaned.matches(".*[0-9]{2,}.*")) {
            if (cleaned.contains("huawei") || cleaned.contains("honor")) {
                return BrandEnum.HUAWEI.getValue();
            }
            if (cleaned.contains("redmi") || cleaned.contains("mi")) {
                return BrandEnum.XIAOMI.getValue();
            }
        }
        return null;
    }

    /**
     * 清洗输入字符串
     */
    private static String cleanInput(String input) {
        return input.toLowerCase()
                // 移除空格
                .replaceAll("\\s+", "")
                // 移除连字符
                .replace("-", "")
                // 移除下划线
                .replace("_", "")
                // 移除点号
                .replace(".", "")
                // 移除中文型号后缀
                .replace("系列", "")
                // 移除中文"手机"字样
                .replace("手机", "");
    }

    static class Trie {
        static class TrieNode {
            Map<Character, TrieNode> children = new HashMap<>();
            BrandEnum brand;
        }

        private final TrieNode root = new TrieNode();

        public void insert(String word, BrandEnum brand) {
            TrieNode node = root;
            for (char c : word.toCharArray()) {
                node = node.children.computeIfAbsent(c, k -> new TrieNode());
            }
            node.brand = brand;
        }

        public BrandEnum searchPrefix(String prefix) {
            TrieNode node = root;
            for (char c : prefix.toCharArray()) {
                node = node.children.get(c);
                if (node == null) {
                    break;
                }
                if (node.brand != null) {
                    return node.brand;
                }
            }
            return null;
        }
    }

    // 测试用例
    public static void main(String[] args) {
        String[] testCases = {
                "XIAOMI", "Redmi Note 12", "POCO F5",
                "OnePlus 11", "realme GT", "HONOR Magic",
                "Galaxy S23", "iPhone 15", "Pixel 8",
                "motorola edge", "nokia 8.3", "unknown-brand"
        };
        for (String brand : testCases) {
            System.out.printf("%-15s -> %s%n", brand, identifyBrand(brand));
        }
    }
}