package com.digiwin.monitor.scan.sdk.util;

import com.digiwin.monitor.scan.sdk.parse.builder.op.OperatorHandlerHolder;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;

/**
 * @description: 扫描class工具类
 * @author: liunansheng
 * @date: 2022/2/18 14:26
 */
public class ClassScanUtil {
    private static final String SLASH = "/";
    private static final String POINT = ".";
    private static final String CLASS_SUFFIX = ".class";

    /**
     * 根据包名扫描包下class名称
     * @param packageName
     * @return
     * @throws IOException
     */
    public static List<String> scanClassName(String packageName) throws IOException {
        String slashedPackageName = StringUtil.replace(packageName, POINT, SLASH);
        URL url = OperatorHandlerHolder.class.getClassLoader().getResource(slashedPackageName);
        String path = getRootPath(url);
        if (isJarFile(path)) {
            return readJarFileClassName(getRootPath(url), slashedPackageName);
        } else {
            return readDirectoryClassName(getRootPath(url), slashedPackageName);
        }
    }

    /**
     * 根据包名扫描包下class
     * @param packageName
     * @return
     * @throws IOException
     */
    public static List<Class> scanClass(String packageName) throws IOException, ClassNotFoundException {
        List<String> classNameList = scanClassName(packageName);
        if (CollectionUtil.isEmpty(classNameList)) {
            return null;
        }
        List<Class> classList = new ArrayList<>();
        for (String className : classNameList) {
            classList.add(Class.forName(className));
        }
        return classList;
    }

    private static List<String> readJarFileClassName(String path, String slashedPackageName) throws IOException {
        JarInputStream jarInputStream = new JarInputStream(new FileInputStream(path));
        JarEntry jarEntry = jarInputStream.getNextJarEntry();
        List<String> nameList = new ArrayList<>();
        while (jarEntry != null) {
            String jarFileName = jarEntry.getName();
            if (jarFileName.startsWith(slashedPackageName) && jarFileName.endsWith(CLASS_SUFFIX)) {
                String className = StringUtil.replace(jarFileName, SLASH, POINT);
                nameList.add(className.substring(0, className.length() - CLASS_SUFFIX.length()));
            }
            jarEntry = jarInputStream.getNextJarEntry();
        }
        return nameList;
    }

    private static List<String> readDirectoryClassName(String path, String slashedPackageName) throws IOException {
        File directory = new File(path);
        File[] childNames = directory.listFiles();
        String packageName = StringUtil.replace(slashedPackageName, SLASH, POINT);
        List<String> nameList = new ArrayList<>();
        for (File file : childNames) {
            String className = packageName + POINT + file.getName();
            nameList.add(className.substring(0, className.length() - CLASS_SUFFIX.length()));
        }
        return nameList;
    }

    private static String getRootPath(URL url) {
        String path = url.getPath();
        int pos = path.indexOf("!");
        if (-1 == pos) {
            return path;
        }
        return path.substring(5, pos);
    }

    private static boolean isJarFile(String path) {
        return path.endsWith(".jar");
    }

}
